#
# This file is part of Rheolef.
#
# Copyright (C) 2000-2009 Pierre Saramito 
#
# Rheolef is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Rheolef is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Rheolef; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------
# ==============================================================================
#
# Textbar
#    used by the 3d fem results viewer
#    gui inspirated from the ghostview gui 
#    - labels (page or data numbers)
#    - mark selectors
#
# author: pierre.saramito@imag.fr
#
# date: 9 may 2001
#
# TODO:
#	- nice colors and sizes
#	- si les noms sont longs, la fenetre ghostview est limitee
#	  et shift-button ou ctrl-button permet de la deplacer
#	  horizontalement, comme la scrollbar !
#       - buttons: unmark all, mark all
# ------------------------------------------------------------------------------
# dependencies:
# 	source tcl_font.tcl
# 	source tcl_percent.tcl
# ==============================================================================

# ------------------------------------------------------------------------------
# ressources:
# ------------------------------------------------------------------------------

option add *Textbar.selectedBackground       #ffff85ef9777  widgetDefault
option add *Textbar.currentBackground        gray70         widgetDefault
option add *Textbar.markedBackground         red            widgetDefault
option add *Textbar.borderwidth              1              widgetDefault; # in pixels

option add *Textbar*fontFamily               helvetica      widgetDefault
option add *Textbar*fontSlant                roman          widgetDefault
option add *Textbar*fontWeight               normal           widgetDefault
option add *Textbar*fontSize                 10             widgetDefault; # in pixel units, not dot units (dpi independent)

# debug
option add *Textbar*textWidthFactor          1              widgetDefault

# in milimeters
option add *Textbar.markerSeparatorHeightMm  2              widgetDefault
option add *Textbar.markerSeparatorWidthMm   2              widgetDefault
option add *Textbar.labelSeparatorWidthMm    2              widgetDefault
option add *Textbar.labelSeparatorHeightMm   2              widgetDefault
option add *Textbar.markerWidthMm            0.5            widgetDefault
option add *Textbar.markerHeightMm           2              widgetDefault
option add *Textbar.textPadxMm               1              widgetDefault
option add *Textbar.textPadyMm               1              widgetDefault

# ==============================================================================
# public member functions
# ==============================================================================

# ------------------------------------------------------------------------------
# usage: textbar_create <win>
# ------------------------------------------------------------------------------
proc textbar_create {win} {
    global textbar_data

    frame $win -class Textbar -borderwidth 0 -relief sunken
    set bg       [option get $win background         Color]
    $win configure -background $bg

    canvas $win.text -width 1 -height 1			\
	-yscrollcommand "$win.bar set" 			\
	-borderwidth 1 -relief sunken

    scrollbar $win.bar -orient vertical -command "$win.text yview"

    pack $win.bar  -side left -expand yes -fill y
    pack $win.text -side left -expand yes -fill y

    set textbar_data($win-init-select-flag) 1
    set textbar_data($win-current-id)  -1
    set textbar_data($win-n-entry)      0
    set textbar_data($win-callback)     ""
    set textbar_data($win-callback-enable) 1
    set textbar_data($win-state)        normal
    set textbar_data($win-height)       0
    set textbar_data($win-font-size)    0
    set textbar_data($win-selected-id)  0
    set textbar_data($win-entry-height) 0
    set textbar_data($win-mark-motion-mode) 0
    set textbar_data($win-mark-motion-initial-position) 0
    set textbar_data($win-mark-motion-previous-position) 0
    set textbar_data($win-mark-motion-previous-direction) 0

    bind $win <Configure> "textbar_redraw  $win"
    bind $win <Destroy>   "textbar_destroy $win"
    return $win
}
# ---------------------------------------------------------------------------
# usage: textbar_bind <win> <event> <command>
# remark: <event> is ignored
# ---------------------------------------------------------------------------
proc textbar_bind {win event cmd} {
    global textbar_data

    if {![info exists textbar_data($win-callback)]} {
        error "bad textbar name \"$win\""
    } else {
        set textbar_data($win-callback) $cmd
    }
}
# ---------------------------------------------------------------------------
# clear all inserted text
# usage: textbar_clear <win>
# ---------------------------------------------------------------------------
proc textbar_clear {win} {
    global textbar_data

    if {![info exists textbar_data($win-callback)]} {
        error "bad textbar name \"$win\""
	return
    }
    $win.text delete all

    set last_id $textbar_data($win-n-entry)
    for {set id 0} {$id < $last_id} {incr id} {
        unset textbar_data($win-text-$id)
        if {[winfo exists $win.text.selected$id]} {
                  destroy $win.text.selected$id
	}
        if {[winfo exists $win.text.marked$id]} {
                  destroy $win.text.marked$id
	}
        if {[winfo exists $win.text.background$id]} {
                  destroy $win.text.background$id
	}
    }
    set textbar_data($win-current-id)  -1
    set textbar_data($win-n-entry)      0
    set textbar_data($win-mark-motion-mode) 0
    set textbar_data($win-mark-motion-initial-position) 0
    set textbar_data($win-mark-motion-previous-position) 0
    set textbar_data($win-mark-motion-previous-direction) 0
}
# ---------------------------------------------------------------------------
# usage: textbar_insert <win> <text>
# ---------------------------------------------------------------------------
proc textbar_insert {win text} {
    global textbar_data

    if {![info exists textbar_data($win-n-entry)]} {
        error "bad textbar name \"$win\""
        return -1
    }
    set id                          $textbar_data($win-n-entry)
    set textbar_data($win-text-$id)       $text
    set textbar_data($win-label2id-$text) $id
    set textbar_data($win-n-entry)        [expr $id + 1]
    # ----------------------------------------
    # fisrt is current
    # ----------------------------------------
    if {$textbar_data($win-selected-id) == -1} {
     set textbar_data($win-selected-id) $id
    }
    set textbar_data($win-mark-$id) $textbar_data($win-init-select-flag)

    # ------------------------
    # dummy bitmap for buttons
    # ------------------------
    image create bitmap nix -data "
        #define nix_width  1
        #define nix_height 1
        static char nix_bits[] = {
            0x00
        };"
    # ------------------------
    # selected:    right buttons, selected page
    # marked:     left button, marked page
    # background: motion, appears active
    # ------------------------
    button $win.text.background$id -image nix
    button $win.text.marked$id     -image nix
    button $win.text.selected$id    -text  $text
}
# ---------------------------------------------------------------------------
# usage: textbar_get <win> <option>
#  options recognized:
#     -marked: returns the list of marked items
# ---------------------------------------------------------------------------
proc textbar_get {win arg} {
    global textbar_data
    if {![info exists textbar_data($win-callback)]} {
        error "textbar_get: bad textbar name \"$win\""
    }
    switch x"$arg" {
        x"-all" {
	    set list {}
    	    set last_id $textbar_data($win-n-entry)
            for {set id 0} {$id < $last_id} {incr id} {
	    	lappend list $textbar_data($win-text-$id)
            }
            return $list
	}
        x"-marked" {
	    set list {}
    	    set last_id $textbar_data($win-n-entry)
            for {set id 0} {$id < $last_id} {incr id} {
		if { $textbar_data($win-mark-$id) } {
		    lappend list $textbar_data($win-text-$id)
		}
            }
            return $list
        }
        default {
            if {![info exists textbar_data($win$arg)]} {
                error "textbar_get: bad field name \"$arg\""
            }
            return $textbar_data(${win}${arg})
        }
    }
}
# ---------------------------------------------------------------------------
# usage: textbar_configure {option value}*
#  options recognized:
#	-select <text>
#	-disable-callback
#	-enable-callback
#  	+ all canvas standard options
# ---------------------------------------------------------------------------
proc textbar_configure {win args} {
    global textbar_data
    if {![info exists textbar_data($win-callback)]} {
        error "textbar_get: bad textbar name \"$win\""
    }
    set std_args ""
    while {[llength $args] > 0} {
        set arg [lindex $args 0]
        set args [lrange $args 1 end]
	#puts "arg $arg"
        switch x"$arg" {
            x"-disable-callback" {
               set textbar_data($win-callback-enable) 0
    	       #puts "disable: callback status: $textbar_data($win-callback-enable)"
               continue
	    }
            x"-enable-callback" {
               set textbar_data($win-callback-enable) 1
    	       #puts "enable: callback status: $textbar_data($win-callback-enable)"
               continue
            }
            x"-select" {
        	    set text  [lindex $args 0]
        	    set args  [lrange $args 1 end]
		    #puts "SELECT \"$text\""
        	    set id $textbar_data(${win}-label2id-${text})
		    #puts "SELECT ID = \"$id\""
        	    textbar_select_button1 ${win} $id
	    }
            default {
                if {![info exists textbar_data(${win}${arg})]} {
        	    set value [lindex $args 0]
        	    set args  [lrange $args 1 end]
        	    set textbar_data(${win}${arg}) $value
                } else {
                    set std_args "$std_args $arg"
	        }
            }
	}
    }
    if {$std_args != ""} {
        eval $win.text configure $std_args
    }
}
# ==============================================================================
# private member functions
# ==============================================================================
proc textbar_destroy {win} {
    global textbar_data

    # -----------
    # delete data
    # -----------
    set last_id $textbar_data($win-n-entry)
    for {set id 0} {$id < $last_id} {incr id} {
        set text $textbar_data($win-text-$id)
        unset textbar_data($win-label2id-$text)
        unset textbar_data($win-text-$id)
        unset textbar_data($win-mark-$id)
    }
    unset textbar_data($win-callback)
    unset textbar_data($win-callback-enable)
    unset textbar_data($win-init-select-flag)
    unset textbar_data($win-current-id)
    unset textbar_data($win-selected-id)
    unset textbar_data($win-n-entry)
    unset textbar_data($win-bitmap)
    unset textbar_data($win-state)
    unset textbar_data($win-height)
    unset textbar_data($win-entry-height)
    unset textbar_data($win-font-size) 
    unset textbar_data($win-mark-motion-mode)
    unset textbar_data($win-mark-motion-initial-position)
    unset textbar_data($win-mark-motion-previous-position)
    unset textbar_data($win-mark-motion-previous-direction)
}
proc textbar_redraw {win} {
    global textbar_data

    $win.text delete all

    set text_width_factor  [option get $win textWidthFactor Color]
    # -----------------
    # find a valid font
    # -----------------
    set ratio       1; # representative size,(when resize, otherwise)
    set family      [option get $win fontFamily Value]
    set slant       [option get $win fontSlant  Value]
    set weight      [option get $win fontWeight Value]
    set fsize       [option get $win fontSize   Value]
    set fsize       [expr $ratio * $fsize]
    set i_fsize     [format %d [expr int($fsize+0.5)]]
    set font        [font_best $family -pxsize $i_fsize -slant $slant -weight $weight]
    set textbar_data($win-font-size) $fsize
    # -----------------
    # colors
    # -----------------
    set fg                          [option get $win foreground Color]
    set bg                          [option get $win background Color]
    set col_selected   		    [option get $win selectedBackground   Color]
    set col_current                [option get $win currentBackground Color]
    set col_marked                  [option get $win markedBackground Color]
    # --------------------
    # recompute parameters
    # --------------------
    set point_per_mm        [expr 72. / 25.4]
    set ratio_mm            [expr $ratio*$point_per_mm]
    set marker_x_sep  	    [expr $ratio_mm * [option get $win markerSeparatorWidthMm  Length]]
    set marker_y_sep        [expr $ratio_mm * [option get $win markerSeparatorHeightMm Length]]
    set text_x_sep          [expr $ratio_mm * [option get $win labelSeparatorWidthMm   Length]]
    set text_y_sep          [expr $ratio_mm * [option get $win labelSeparatorHeightMm  Length]]
    set marker_width        [expr $ratio_mm * [option get $win markerWidthMm  Length]]
    set marker_height       [expr $ratio_mm * [option get $win markerHeightMm  Length]]
    set text_padx           [expr $ratio_mm * [option get $win textPadxMm  Length]]
    set text_pady           [expr $ratio_mm * [option get $win textPadxMm  Length]]

    # ---------------------------------
    # compute max text width
    # ---------------------------------
    set max_text_width 0
    set max_text_length 0
    set last_id $textbar_data($win-n-entry)
    for {set id 0} {$id < $last_id} {incr id} {
    
        set entry $textbar_data($win-text-$id)
        set text_width [font measure $font -displayof $win.text $entry]
        set text_width [expr $text_width_factor*$text_width]
        if {$max_text_width < $text_width} {
	     set max_text_width $text_width
        }
        set text_length [string length $textbar_data($win-text-$id)]
        if {$max_text_length < $text_length} {
	     set max_text_length $text_length
        }
    }
    # ---------------------------------
    # place marker and text
    # ---------------------------------
    set xstart 0
    set x0  $xstart
    set xm0 [expr $xstart + $marker_x_sep]
    set xm1 [expr $xm0 + $marker_width ]
    set xt0 [expr $xm1 + $marker_x_sep + $text_x_sep]
    set xt1 [expr $xt0 + $max_text_width + 2*$text_padx]
    set x1  [expr $xt1 + $text_x_sep ]
    set width  [expr $x1]

    set text_height [expr $fsize]
    set text_dy   [expr $text_height   + 2.0*($text_y_sep + $text_pady)]
    set marker_dy [expr $marker_height + 2.0*$marker_y_sep ]
    if {$marker_dy > $text_dy } { set dy $marker_dy } else { set dy $text_dy }
    set ystart 0
    set y0  $ystart
    set y2  [expr $y0 + 0.5*double($dy)]
    set ym0 [expr $y2 - 0.5*double($marker_height)]
    set ym1 [expr $y2 + 0.5*double($marker_height)]
    set yt0 [expr $y2 - 0.5*double($text_height+2*$text_pady)]
    set yt1 [expr $y2 + 0.5*double($text_height+2*$text_pady)]
    set y1  [expr $y0 + $dy]

    set active_borderwidth  [option get $win borderwidth Length]

    set last_id $textbar_data($win-n-entry)
    for {set id 0} {$id < $last_id} {incr id} {
 
        set shift   [expr $id*$dy] 
        set ymiddle [expr $shift + 0.5*$dy] 
        # -------------------------------------
        # background button
        # ------------------------------------
	set yib0 [expr $y0   + $shift] 
	set yib1 [expr $yib0 + $dy] 
        
	if { $textbar_data($win-current-id) == $id } {
	    set color  $col_current
	    set relief sunken
	    set borderwidth $active_borderwidth
	} else {
	    set color  $bg
	    set relief flat
	    set borderwidth 0
        }
        $win.text.background$id configure 			\
	-width 			$width				\
	-height 		$dy				\
	-state			disabled			\
	-relief			$relief				\
	-borderwidth 		$borderwidth			\
	-foreground		$fg				\
	-activebackground  	$color				\
	-activeforeground  	$color				\
	-foreground	  	$color				\
	-highlightbackground  	$color				\
	-highlightcolor  	$color				\
	-background		$color	


        $win.text create window 				\
				$x0				\
				$ymiddle			\
	-width 			$width				\
	-height 		$dy		 	        \
	-window 		$win.text.background$id		\
	-anchor			w				\
        -tags 			[ list sensor$id ]

        # -------------------------------------
        # selected button = background for text
        # ------------------------------------
	set yit0 [expr $yt0  + $shift] 
	set yit1 [expr $yit0 + $text_height] 
	if {$textbar_data($win-selected-id) == $id } {
	    set color  $col_selected
	    set relief sunken
	    set borderwidth $active_borderwidth
	} else {
	    set color  $bg
	    set relief flat
	    set borderwidth 0
        }
        $win.text.selected$id configure 				\
	-width 			$max_text_length		\
	-font			$font				\
	-justify	  	right				\
	-state			disabled			\
	-relief			$relief				\
	-borderwidth 		$borderwidth			\
	-padx	 		$text_padx			\
	-pady	 		$text_pady			\
	-disabledforeground  	$fg				\
	-foreground  		$color				\
	-activebackground  	$color				\
	-activeforeground  	$color				\
	-foreground	  	$color				\
	-highlightbackground  	$color				\
	-highlightcolor  	$color				\
	-background		$color	

        $win.text create window 				\
				$xt0				\
				$ymiddle			\
	-window 		$win.text.selected$id		\
	-anchor 		w				\
        -tags 			[ list sensor$id ]

	# -width 			$max_text_width			\
	# -height 		$text_height			\
        # ------------------------
        # marker = current pages
        # ------------------------
	set yim0 [expr $ym0  + $shift] 
	set yim1 [expr $yim0 + $marker_height] 


        if {$textbar_data($win-mark-$id)} {
	    set relief sunken
	    set color  $col_marked
	    set borderwidth $active_borderwidth
	} else {
	    set relief flat
	    set color  $bg
	    set borderwidth 0
	}
        $win.text.marked$id configure 				\
	-width 			$marker_width			\
	-height 		$marker_height			\
	-state			disabled			\
	-relief			$relief				\
	-borderwidth 		$borderwidth			\
	-foreground		$fg				\
	-activebackground  	$color				\
	-activeforeground  	$color				\
	-foreground	  	$color				\
	-highlightbackground  	$color				\
	-highlightcolor  	$color				\
	-background		$color


        $win.text create window 				\
				$xm0				\
				$ymiddle			\
	-window 		$win.text.marked$id		\
	-anchor			w				\
        -tags 			[ list sensor$id ]
        # ---------------------------------
        # binding: put buttons instead
        # ---------------------------------
        bind $win.text.background$id <ButtonPress>   "textbar_select  $win $id %b %y $win.text.background$id"
        bind $win.text.background$id <ButtonRelease> "textbar_release $win $id %b %y $win.text.background$id"
        bind $win.text.background$id <Motion>        "textbar_motion  $win $id %b %y $win.text.background$id"
        bind $win.text.background$id <Enter>         "textbar_enter   $win $id %b"
        bind $win.text.background$id <Leave>         "textbar_leave   $win $id %b"

        bind $win.text.selected$id <ButtonPress>     "textbar_select  $win $id %b %y $win.text.selected$id"
        bind $win.text.selected$id <ButtonRelease>   "textbar_release $win $id %b %y $win.text.selected$id"
        bind $win.text.selected$id <Motion>          "textbar_motion  $win $id %b %y $win.text.selected$id"
        bind $win.text.selected$id <Enter>           "textbar_enter   $win $id %b"
        bind $win.text.selected$id <Leave>           "textbar_leave   $win $id %b"

        bind $win.text.marked$id <ButtonPress>       "textbar_select  $win $id %b %y $win.text.marked$id"
        bind $win.text.marked$id <ButtonRelease>     "textbar_release $win $id %b %y $win.text.marked$id"
        bind $win.text.marked$id <Motion>            "textbar_motion  $win $id %b %y $win.text.marked$id"
        bind $win.text.marked$id <Enter>             "textbar_enter   $win $id %b"
        bind $win.text.marked$id <Leave>             "textbar_leave   $win $id %b"
    }
    # resize in width and update sizes for scrollbar:
    set height [expr $last_id*$dy]
    $win.text configure -width $width -scrollregion [list 0 0 $width $height]
    set textbar_data($win-height)       $height
    set textbar_data($win-entry-height)	$dy
}
proc textbar_leave_raw {win id} {

    global textbar_data
    set textbar_data($win-current-id) -1

    set bg  [option get $win background Color]
    $win.text.background$id configure 				\
	-relief			flat				\
	-borderwidth 		0				\
	-activebackground  	$bg				\
	-activeforeground  	$bg				\
	-foreground	  	$bg				\
	-highlightbackground  	$bg				\
	-highlightcolor  	$bg				\
	-background		$bg	

    if { $textbar_data($win-selected-id) != $id } {

        $win.text.selected$id configure 			\
	-activebackground  	$bg 				\
	-activeforeground  	$bg 				\
	-foreground	  	$bg 				\
	-highlightbackground  	$bg 				\
	-highlightcolor  	$bg 				\
	-background		$bg
    }
    if { $textbar_data($win-mark-$id) == 0 } {

        $win.text.marked$id configure 				\
	-relief			flat				\
	-borderwidth 		0				\
	-activebackground  	$bg				\
	-activeforeground  	$bg				\
	-foreground	  	$bg				\
	-highlightbackground  	$bg				\
	-highlightcolor  	$bg				\
	-background		$bg
    }
}
proc textbar_leave {win id button} {

    global textbar_data
    if { $textbar_data($win-mark-motion-mode) != 0 } {
   	return
    } 
    textbar_leave_raw $win $id
}
proc textbar_enter_raw {win id} {

    global textbar_data
    set textbar_data($win-current-id) $id

    set col_current         [option get $win currentBackground   Color]
    set active_borderwidth  [option get $win borderwidth Length]
    set color               [option get $win background Color]

    $win.text.background$id configure 				\
	-relief			sunken				\
	-borderwidth 		$active_borderwidth		\
	-activebackground  	$col_current			\
	-activeforeground  	$col_current			\
	-foreground	  	$col_current			\
	-highlightbackground  	$col_current			\
	-highlightcolor  	$col_current			\
	-background		$col_current	
    
    if { $textbar_data($win-selected-id) != $id } {

        $win.text.selected$id configure 			\
	-activebackground  	$col_current			\
	-activeforeground  	$col_current			\
	-foreground	  	$col_current			\
	-highlightbackground  	$col_current			\
	-highlightcolor  	$col_current			\
	-background		$col_current
    }
    if { $textbar_data($win-mark-$id) == 0 } {

        $win.text.marked$id configure 				\
	-relief			flat				\
	-borderwidth 		0				\
	-activebackground  	$col_current			\
	-activeforeground  	$col_current			\
	-foreground	  	$col_current			\
	-highlightbackground  	$col_current			\
	-highlightcolor  	$col_current			\
	-background		$col_current
    }
}
proc textbar_enter {win id button} {

    global textbar_data
    if { $textbar_data($win-mark-motion-mode) != 0 } {
   	return
    }
    textbar_enter_raw $win $id
} 
proc textbar_select_button1 {win id} {
    global textbar_data

    set prev_id                        $textbar_data($win-selected-id)
    set textbar_data($win-selected-id) $id

    set bg          	    [option get $win background Color]
    set col_selected 	    [option get $win selectedBackground Color]
    set active_borderwidth  [option get $win borderwidth Length]

    if { $prev_id != -1 } {
        $win.text.selected${prev_id} configure 			\
	-relief			flat				\
	-borderwidth 		0				\
	-activebackground  	$bg				\
	-activeforeground  	$bg				\
	-foreground	  	$bg				\
	-highlightbackground  	$bg				\
	-highlightcolor  	$bg				\
	-background		$bg
    }
    $win.text.selected${id} configure 				\
	-relief			sunken				\
	-borderwidth 		$active_borderwidth		\
	-activebackground  	$col_selected			\
	-activeforeground  	$col_selected			\
	-foreground	  	$col_selected			\
	-highlightbackground  	$col_selected			\
	-highlightcolor  	$col_selected			\
	-background		$col_selected

    # -----------------
    # call back, if any
    # -----------------
    #puts "callback status: $textbar_data($win-callback-enable)"
    if {$textbar_data($win-callback-enable) && \
        [string trim $textbar_data($win-callback)] != ""} {
        set cmd      $textbar_data($win-callback)
        set value    $textbar_data($win-text-$id)
        set cmd      [percent_subst %v $cmd $value]
        uplevel #0 $cmd
    }
}
proc textbar_swap_mark {win id} {
    global textbar_data

    if { $textbar_data($win-mark-$id) } {
	set textbar_data($win-mark-$id) 0
        if { $textbar_data($win-current-id) != $id } {
            set color [option get $win background Color]
	} else {
            set color [option get $win currentBackground   Color]
        }
        $win.text.marked$id configure 				\
	-relief			flat				\
	-borderwidth 		0				\
	-activebackground  	$color				\
	-activeforeground  	$color				\
	-foreground	  	$color				\
	-highlightbackground  	$color				\
	-highlightcolor  	$color				\
	-background		$color

    } else {
	set textbar_data($win-mark-$id) 1
        set col_marked          [option get $win markedBackground Color]
        set active_borderwidth  [option get $win borderwidth      Length]
        $win.text.marked$id configure 				\
	-relief			sunken				\
	-borderwidth 		$active_borderwidth		\
	-activebackground  	$col_marked			\
	-activeforeground  	$col_marked			\
	-foreground	  	$col_marked			\
	-highlightbackground  	$col_marked			\
	-highlightcolor  	$col_marked			\
	-background		$col_marked
    }
}
proc textbar_select_button2 {win id y} {
    global textbar_data
    set textbar_data($win-mark-motion-initial-position)   $y
    set textbar_data($win-mark-motion-previous-position)  $y
    set textbar_data($win-mark-motion-previous-direction) 0
    set textbar_data($win-mark-motion-mode)               1
    textbar_swap_mark $win $id
}
proc textbar_rely_to_y {win rely relwin} {
    global textbar_data
    set height $textbar_data($win-height)
    set alpha [lindex [$win.text yview] 0]
    set y0    [expr $alpha * $height]
    set ybase [winfo y $relwin]
    set y     [expr $y0 + $ybase + $rely]
    return $y
}
proc textbar_select {win id button rely relwin} {

    global textbar_data

    if { $button == 1 } {
	textbar_select_button1 $win $id
	return
    }
    if { $button == 2 } {
        set textbar_data($win-mark-motion-mode) 1
        set y [textbar_rely_to_y $win $rely $relwin]
	textbar_select_button2 $win $id $y
	return
    }
}
proc textbar_pos_to_id {win y} {
    global textbar_data
    set dy $textbar_data($win-entry-height)
    if { $dy == 0 } {
	# parano setting !
	return 0
    }
    set id [expr int(double($y)/$dy)]
    if { $id < 0 } { 
    	return 0
    }
    set last_id [expr $textbar_data($win-n-entry) - 1]
    if { $id > $last_id } {
	return $last_id
    }
    return $id
}
# swap in range [start, last[
proc textbar_swap_mark_interval {win start_id last_id dir} {

    for {set id $start_id} {$id != $last_id} {incr id $dir} {
        textbar_swap_mark $win $id
    }
}
proc textbar_motion {win id button rely relwin} {

    global textbar_data
    if { $textbar_data($win-mark-motion-mode) == 0 } {
        return
    }
    set prev_y   $textbar_data($win-mark-motion-previous-position)
    set y        [textbar_rely_to_y $win $rely $relwin]
    set prev_id  [textbar_pos_to_id $win $prev_y]
    set id       [textbar_pos_to_id $win $y]
    set prev_dir $textbar_data($win-mark-motion-previous-direction)

    if { $y >= $prev_y } {
	set dir 1
    } elseif { $y < $prev_y } {
        set dir -1
    } else {
        set dir $prev_dir
    }
    if { $prev_dir == 0 } {
      set prev_dir $dir
    }
    set textbar_data($win-mark-motion-previous-position)  $y
    set textbar_data($win-mark-motion-previous-direction) $dir

    if { $prev_dir == $dir && $prev_id == $id } {
	# no change
   	return
    }
    if { $prev_id != $id } {
	textbar_leave_raw $win $prev_id
	textbar_enter_raw $win $id
    }
    if { $prev_dir == $dir } {
	# start from the next entry
        incr prev_id $dir
        set n_entry $textbar_data($win-n-entry)
	if { $prev_id < 0 } {
	    set prev_id 0
	} elseif { $prev_id >= $n_entry } {
	    set prev_id  [expr $n_entry - 1]
	}
    }
    set last_id [expr $id + $dir]
    textbar_swap_mark_interval $win $prev_id $last_id $dir
}
proc textbar_release {win id button rely relwin} {

    if { $button != 2 } { return }
    global textbar_data
    #set y [textbar_rely_to_y $win $rely $relwin]
    #textbar_motion $win $id $button $y
    set textbar_data($win-mark-motion-mode) 0
}
