/*
 * Picviz - Parallel coordinates ploter
 * Copyright (C) 2008 Sebastien Tricaud <toady@gscore.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $Id: image.c 320 2008-11-18 07:16:04Z toady $
 */

#include <stdio.h>
#include <stdlib.h>

#include <picviz.h>

/**
 * \ingroup PicvizMain
 * @{
 */

/** \file image.c
 * \brief Create and manipulate a parallel coordinates image
 */

struct engine_t engine;

/**
 * Creates a new parallel coordinates empty image
 *
 * \return the new picviz image or NULL on error
 */
PicvizImage *picviz_image_new(void)
{
	PicvizImage *pcimage = NULL;

	pcimage = malloc(sizeof(*pcimage));
	if ( ! pcimage ) {
		fprintf(stderr, "Cannot create image: memory exhausted.\n");
		return NULL;
	}
	pcimage->height        = engine.image_height;
	//pcimage->header_height = DEFAULT_IMAGE_HEADER_HEIGHT;
	pcimage->header_height = pcimage->height / engine.font_factor + 5;
	pcimage->width         = 0;
	pcimage->filter        = NULL;
	pcimage->zero_pos      = TOP;
	pcimage->title         = "";
	pcimage->bgcolor       = "#FFFFFF"; /* White background default */
	pcimage->lines_max     = 0;

	INIT_LLIST_HEAD(&pcimage->axes);
	INIT_LLIST_HEAD(&pcimage->lines);

	picviz_correlation_new(&pcimage->correlation);

	return pcimage;
}

void picviz_image_destroy(PicvizImage *image)
{
	PicvizAxis *axe, *axebkp;
	struct line_t *line, *linebkp;

	llist_for_each_entry_safe(axe, axebkp, &image->axes, list)
		picviz_axis_destroy(axe);

	llist_for_each_entry_safe(line, linebkp, &image->lines, list)
		picviz_line_free(line);

	picviz_correlation_destroy(image->correlation);

	free(image);
}

/**
 * Increases image width
 *
 * \param i the image to update
 * \param expand width value to increase
 */
void picviz_image_width_increase(PicvizImage *i, unsigned int expand)
{
        i->width += expand;
}


void picviz_image_axis_append(PicvizImage *i, PicvizAxis *axis)
{
        picviz_image_width_increase(i, engine.axis_default_space);
        axis->xpos = engine.axis_x_cursor;
        engine.axis_x_cursor += engine.axis_default_space;
        llist_add_tail(&axis->list, &i->axes);
}

void picviz_image_line_append(PicvizImage *image, struct line_t *line)
{
        llist_add_tail(&line->list, &image->lines);
	image->lines_max++;
}

void picviz_image_debug_printall(PicvizImage *i)
{
        PicvizAxis *a;
        struct line_t *l;
        struct axisplot_t *axisplot;

        printf("i->width=%d\n", i->width);
        printf("i->height=%lld\n", i->height);
        printf("i->header_height=%lld\n", i->header_height);
        printf("i->zero_pos=%d\n", i->zero_pos);
        printf("i->bgcolor=%s\n", i->bgcolor);
        llist_for_each_entry(a, &i->axes, list) {
                printf("    a->id=%llu\n", a->id);
                printf("    a->label=%s\n", picviz_properties_get(a->props, "label"));
                printf("    a->type=%d\n", a->type);
                printf("    a->xpos=%d\n", a->xpos);
                printf("\n");
        }
        llist_for_each_entry(l, &i->lines, list) {
                printf("l->id=%llu\n", l->id);
                printf("l->props->color=%s\n", picviz_properties_get(l->props, "color"));
                llist_for_each_entry(axisplot, &l->axisplot, list) {
                        printf("    axisplot->strval=%s\n", axisplot->strval);
                        printf("    axisplot->y=%lld\n", axisplot->y);
                        printf("    axisplot->axis_id=%llu\n", axisplot->axis_id);
                }
        }
}

void picviz_init(void)
{

	engine_init();

	//fprintf(stdout, "Picviz - (c) 2008 Sebastien Tricaud\n");
	//fprintf(stdout, "[+] Loaded render plugins:\n");
	//fprintf(stdout, "[+] Loaded output plugins: svg\n");

}

#ifdef _UNIT_TEST_
int main(void)
{
        PicvizImage *image;
        PicvizAxis *axis;
        struct line_t *line;
        int i = 0;
        float value;

        picviz_init();
        image = picviz_image_new();
        axis = picviz_axis_init();
        picviz_image_axis_append(image, axis);
        picviz_axis_set_type_from_string(axis, "string");
        value = picviz_line_value_get_from_string(axis, NULL, "foo");
        printf("value=%f\n", value);
        picviz_axis_prop_set_label(axis, "foo");
        line = picviz_line_init();
        picviz_axis_line_append(axis, line);
        axis = picviz_axis_init();
        picviz_image_axis_append(image, axis);
        picviz_axis_set_type_from_string(axis, "integer");
        picviz_axis_prop_set_label(axis, "bar");
        line = picviz_line_init();
        picviz_axis_line_append(axis, line);
        line = picviz_line_init();
        picviz_axis_line_append(axis, line);

        picviz_image_debug_printall(image);

}
#endif

/**
 * @}
 */

